#!/usr/bin/python
# Writes individual sectors from stdin to disk.
# 
# Intended to be used with an mbed to interface to the HP 4145A's disk controller.
# See http://www.kitsune-denshi.net/equipment:hp4145a for a description

import serial
import sys
import time

with serial.Serial('/dev/ttyACM0', 115200, timeout=5) as ser:


    print("Initialising...", file = sys.stderr)

    ser.reset_input_buffer()

    #release controller from reset
    ser.write(b"1\n")
    time.sleep(0.1)

    #select controller
    ser.write(b"4\n")
    time.sleep(0.1)

    #enable drive
    ser.write(b"5\n")
    time.sleep(1)

    #reset drive
    ser.write(b"w\n00 03\n")  
    time.sleep(5)

    #read status
    ser.write(b"r\n00\n")  

    print("Status register = {}".format(ser.readline().decode("utf-8").rstrip()), file = sys.stderr)


    prev_track = -1

    for line in sys.stdin


        temp = line.rstrip().split(':')
        track = int(temp[0])
        sector = int(temp[1])

        #seek to track if required 
        if not (track == prev_track):
            ser.write(bytes("w\n03 {:02X}\n".format(track), "utf-8"))
            ser.write(b"w\n00 1B\n")
            #a bit short, but let's assume the tracks are in order
            time.sleep(0.5)

        prev_track = track

        print("Writing track {}, sector {} ".format(track, sector), file = sys.stderr)


        #load sector number
        ser.write(bytes("w\n02 {:02X}\n".format(sector), "utf-8"))

        #clear buffer
        ser.write(b"c\n")

        #transfer data
        values = temp[2].split(' ')
        for v in values:
            ser.write(bytes("i\n" + v + "\n", "utf-8"))

        #read back buffer
        ser.write(b"o\n")        
        print("{:02d}:{:02d}:".format(track, sector) + ser.readline().decode("utf-8").rstrip())

        for n in range(0,1):
            #write sector
            ser.write(b"P\n")
            bytes_read = int(ser.readline().decode("utf-8").rstrip())
            print("{} bytes transferred from buffer.".format(bytes_read), file = sys.stderr)

            #read status
            ser.write(b"r\n00\n")  
            print("Status register = {}".format(ser.readline().decode("utf-8").rstrip()), file = sys.stderr)

            time.sleep(0.1)

        #clear buffer
        ser.write(b"c\n")

        #read sector
        ser.write(b"Q\n")
        bytes_read = int(ser.readline().decode("utf-8").rstrip())
        print("{} bytes read back.".format(bytes_read), file = sys.stderr)

        #read status
        ser.write(b"r\n00\n")  
        print("Status register = {}".format(ser.readline().decode("utf-8").rstrip()), file = sys.stderr)

        #read back buffer
        ser.write(b"o\n")        
        print("{:02d}:{:02d}:".format(track, sector) + ser.readline().decode("utf-8").rstrip()) 


    #reset drive
    ser.write(b"w\n00 03\n")  
    time.sleep(5)

    #disable drive
    ser.write(b"6\n")

    #turn off controller
    ser.write(b"2\n")
    ser.write(b"3\n")









