/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * 
 * Agilent / Keysight U1273A display interface adapter
 * 
 * ICE5LP1K HDL code v0.1
 *
 * http://www.kitsune-denshi.net
 *
 * See LICENSE.txt
 *
 * Version history:
 * v0.1  2021-02-12 initial relase, for R0 electronics design, limited testing
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */


/* 
 * displayif:
 * 
 * translates commands from U1273A multrimeter to SSD1309 OLED controller commands:
 * - A0 command is changed to  A1 to mirror display
 * - column addresses are shifted by -2
 * - data written to columns 0 or 1 are suppressed
 * - data written after command to initialise bottom 8 rows are suppressed (B0 01 1F)
 *
 * known issues:
 * - will not correct for wrapping on upper memory boundary
 * - reading from display are not supported
 * - relies on some aspects speciric to how the  U1273A drives the dusplay, so may not work universally
 */


module displayif(
  input [7:0] in_data,          //data input from multimeter
  input in_res,                 //reset input from multimeter
  input in_read,                //~RD signal from multimeter
  input in_write,               //~WR signal from multimeter
  input in_dc,                  //D/~C signal from multimeter
  output reg [7:0] out_data,    // data output to display
  output wire out_res,          //reset output to display
  output wire out_read,         //~RD signal to display
  output wire out_write,        //~WR signal to display
  output reg out_dc,            //D/~C signal to display
  output wire out_cs,           //~CS signal to display
  output wire [1:0] debug       //FPGA debug output
);
  reg ovf;                      //carry from column address lower 4 bits - 2 (if set, also decrement address higher 4 bits)
  reg sub_inh;                  //set if 2-byte command start detected and inhibits subtraction from address (prevents erroneous changes to 2nd command byte)
  reg [1:0] addr_ovr;           //keeps track of number of bytes to suppress when column 0 or 1 addressed
  reg [1:0] ll_state;           //state for tracking B0 01 1F command sequence
  reg addr_lsb;                 //LSB of address lower 4 bits (needed to determine if columns 0 or 1 are addressed when looking at upper 4 bits)

  
  always @(*) begin
   
    //changes to command bytes
    if(in_dc == 1'b0) begin
      out_dc = 1'b0;
      
      casez(in_data)
      
        //0X - lower column address: subtract 2 unless it's part of a 2-byte command
        8'b0000????: begin
          if(!sub_inh) begin
            out_data[3:0] = in_data[3:0] - 2;
            out_data[7:4] = in_data[7:4];
          end 
          else begin
            out_data = in_data;
          end
        end
        
        //1X - higher column address: subtract 1 if carry from lower 4 bits, but do not underflow 
        8'b0001????: begin 
          if(ovf && (in_data != 8'h10)) begin
            out_data[3:0] = in_data[3:0] - 1;
            out_data[7:4] = in_data[7:4];
          end 
          else begin
            out_data = in_data;
          end
        end
        
        // A0 / A1 - segment re-map to (un-)mirror display
        8'b1010000?: begin
          out_data = 8'hA1;
        end

        //leave all other commands as they are
        default: begin
          out_data = in_data;
        end
      endcase

    end 
      
    //changes to data bytes
    else begin
    
      //if data is being written to columns 0 or 1, output address 0 command instead (so subsequent data will start correctly at column 0)
      if(addr_ovr) begin                
        out_data = 8'h00;
        out_dc = 1'b0;
      end
      else begin
        //if last 8 rows are being initialised, output 00 as data
        if(ll_state == 2'b11) begin
          out_data = 8'h00;
        end 
        //leave all other data as they are
        else begin
          out_data = in_data;
        end
        out_dc = 1'b1;

      end            
    end
      
  end //asynchronous block
    
  always @(posedge in_write or negedge in_res) begin

    //asynchronous reset for registers
    if(!in_res) begin
        ovf <= 1'b0;
        sub_inh <= 1'b0;
        addr_ovr <= 2'b0;
        ll_state <= 2'b0;
        addr_lsb <= 1'b0;
    end 
    else begin
      
      //set overflow flag if lower 4 address bits - 2 would output a carry 
      //(i.e. ovf will store whether the lower address bits were 0000 or 0001)
      //also save LSB separately
      if((in_data[7:1]==7'b0000000) && (in_dc == 1'b0) && (sub_inh == 1'b0)) begin
        ovf <= 1'b1;
        addr_lsb <= in_data[0];
      end
      else begin
        ovf <= 1'b0;
      end        
      
      //detect whether column 0 or 1 is being addressed 
      //the ovf flag is used to check whether previously seen lower 4 address bits were 0000 or 0001
      //if column 0 or 1 is being addressed, set how many data bytes to suppress based on the LSB from the lower address
      if(ovf && (in_dc == 1'b0) && (in_data == 8'h10)) begin
        if(addr_lsb) begin
            addr_ovr <= 2'b01;
         end
         else begin
            addr_ovr <= 2'b10;
         end                
      end
      
      //decrement the number of bytes to left to suppress in columns 0 or 1 as each byte is suppressed
      if(in_dc == 1'b1 && addr_ovr) begin
        addr_ovr <= addr_ovr - 1;
      end
      
      //detect start bytes of 2-byte command sequences to prevent the second byte from being interpreted as an address
      if(in_dc == 1'b0 && (in_data == 8'hD3 || in_data == 8'hD8)) begin
        sub_inh <= 1'b1;
      end
      else begin
        sub_inh <= 1'b0;
      end
      
      //state machine for detecing the command sequence B0 01 1F
      //ll_state keeps track of how many bytes of the sequence have been detected (11 meaning fully detected)
      //once the sequence has been detected, ll_state will remain in 11 as long as data bytes are output and will be reset with the next command byte
      case(ll_state)

        2'b00: begin
          if((!in_dc) && (in_data == 8'hB0)) begin
            ll_state <= 2'b01;
          end
          else begin
            ll_state <= 2'b00;
          end
        end
        
        2'b01: begin
          if((!in_dc) && (in_data == 8'h01)) begin
            ll_state <= 2'b10;
          end
          else begin
            ll_state <= 2'b00;
          end
        end
        
        2'b10: begin
          if((!in_dc) && (in_data == 8'h1F)) begin
            ll_state <= 2'b11;
          end
          else begin
            ll_state <= 2'b00;
          end
        end
        
        2'b11: begin
          if(in_dc) begin
            ll_state <= 2'b11;
          end
          else begin
            if(in_data == 8'hB0) begin
              ll_state <= 2'b01;
            end
            else begin
              ll_state <= 2'b00;
            end
          end
        end
      endcase
      
    end //if not reset
      
  end //synchronous block
  

  assign debug = addr_ovr;
  assign out_res = in_res;
  assign out_read = 1'b1;
  assign out_write = in_write;
  assign out_cs = 1'b0;

endmodule